#!/usr/bin/perl

use strict;
use warnings;

use Time::HiRes qw(gettimeofday tv_interval);
use DBI;
use Getopt::Long;
use File::Basename;

# Constants and Globals
# ---------------------

our $MyName = basename($0);
my $lRuns = 10000;

# Functions
# ---------

sub Usage
{
  print("
SYNOPSIS

  $MyName flags

DESCRIPTION

  Shows the impact of bigger transactions on performance.

FLAGS

  --create, -c    Create table for test.
  --database, -D  Database to use [test].
  --drop, -d      Drop table after test.
  --inserts, -i   # of inserts per commit [1].
  --help, -?      Display this help and exit.
  --host, -h      Connect to host [localhost].
  --password, -p  Password to use when connecting to server [].
  --port, -P      Port number to use for connection [3306].
  --socket, -S    Socket file to use for connection [/var/mysql/mysql.sock].
  --user, -u      User for login [test].

PARAMETERS

  none

");
}

# Process parameters
# ------------------

my $optCreate = 0;
my $optDatabase = 'test';
my $optDrop = 0;
my $optInserts = 1;
my $optHelp = 0;
my $optHost = 'localhost';
my $optPassword = '';
my $optPort = 3306;
my $optSocket = '/var/mysql/mysql.sock';
my $optUser = 'test';

my $rc = GetOptions( 'create|c' => \$optCreate
                   , 'database|D=s' => \$optDatabase
                   , 'drop|d' => \$optDrop
                   , 'inserts|i=i' => \$optInserts
                   , 'help|?' => \$optHelp
                   , 'host|h=s' => \$optHost
                   , 'password|p=s' => \$optPassword
                   , 'port|P=i' => \$optPort
                   , 'socket|S=s' => \$optSocket
                   , 'user|u=s' => \$optUser
                   );

if ( $optHelp ) {
    &Usage();
    exit(0);
}

if ( ! $rc) {
    &Usage();
    exit(1);
}

if(@ARGV != 0) {
    &Usage();
    exit(2);
}

# Check parameter
# ---------------

my $commit_every = $optInserts;
my ($lUser, $lPassword, $lHost, $lDatabase) = ($optUser, $optPassword, $optHost, $optDatabase);

# Start here
# ----------

# open connection

my ($dbh, $sql, $sth);

$dbh = DBI->connect("DBI:mysql:$lDatabase:$lHost:$optPort"
                    , $lUser, $lPassword
                    , { RaiseError => 0
                      , PrintError => 0
                      , AutoCommit => 0
                      }
                    );

if ( $DBI::err ) {
  print "Error opening DB: " . DBI::errstr() . "\n";
  exit(1);
}

$dbh->{AutoCommit} = 0;

# create the table

if ( $optCreate ) {

  $sql = "CREATE TABLE " . $optDatabase . ".commit_demo
(
    id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY
) ENGINE=InnoDB";
  $dbh->do($sql);
  $dbh->disconnect;
  exit(0);
}

# drop the table

if ( $optDrop ) {

  $sql = "DROP TABLE " . $optDatabase . ".commit_demo";
  $dbh->do($sql);
  $dbh->disconnect;
  exit(0);
}

# run the test

print "Truncate table.\n";

$sql = "TRUNCATE TABLE " . $optDatabase . ".commit_demo";
$dbh->do($sql);

my $t0 = [gettimeofday];

$sql = "INSERT INTO " . $optDatabase . ".commit_demo (id)
  VALUES (?)";
$sth= $dbh->prepare($sql);
if ( $sth->err ) {
  print "ERROR execute: " . $sth->errstr();
  exit(2);
}

$| = 1;   # Write print output synchronous

my $i = 1;
while ( $i <= $lRuns ) {

  $sth->execute('NULL');
  if ( $sth->err ) {
    print "ERROR execute: " . $sth->errstr();
    exit(2);
  }

  if ( $i % $commit_every == 0 ) {
    print ".";
    $dbh->commit();
  }

  $i++;
}
$dbh->commit();
$sth->finish;

my $t1 = [gettimeofday];
print "\nDuration: " . tv_interval($t0, $t1) . " seconds.\n";

$dbh->disconnect;
exit(0);
