#!/usr/bin/perl -w

use strict;
use warnings;

use Getopt::Long;
use File::Basename;
use Time::Local;

# Constants and Globals
# ---------------------

our $lMyName = basename($0);

# Functions
# ---------

# ------------------------------------------------------------------------------
sub Usage
# ------------------------------------------------------------------------------
{
  print <<EOF;
SYNOPSIS

  $lMyName flags

DESCRIPTION

  Creates a runnable benchmark script from the MySQL general log.

FLAGS

  help, h          Print this help.
  debug            Enable debug mode.
  with-thinktimes  Creates "think-times". The wrong assumption is made
                   that a query runs less than 1 second!
  without-dml      Excludes all DML commands (INSERT, UPDATE, DELETE,
                   REPLACE).
  general-log      The general log to parse.

PARAMETERS

  none

EOF
}

# Process parameters
# ------------------

my $oHelp = 0;
my $oDebug = 0;
my $oWithThinkTimes = 0;
my $oWithoutDml = 0;
my $oGeneralLog = `hostname` . '.log';

my $rc = GetOptions( 'help|h|?' => \$oHelp
                   , 'debug' => \$oDebug
                   , 'with-thinktimes' => \$oWithThinkTimes
                   , 'without-dml' => \$oWithoutDml
                   , 'general-log=s' => \$oGeneralLog
                   );

if ( $oHelp ) {
    &Usage();
    exit(0);

}

if ( ! $rc) {
    &Usage();
    exit(1);
}

if(@ARGV != 0) {
    &Usage();
    exit(2);
}

# Start here
# ----------

if ( ! defined($oGeneralLog) ) {
  print("Error: no filename\n");
  exit 1;
}

if ( ! -r $oGeneralLog ) {
  print("Cannot read file $oGeneralLog\n");
  exit 1;
}

my $lWriteSemicolon = 0;
my $lWeAreInDmlQueryNow = 0;
my ($lOldTimestamp, $lNewTimestamp) = (0, 0);   # For calculating sleep time
open INPUT, "<$oGeneralLog";
while ( <INPUT> ) {

  chomp;

  # skip 3 intro lines after restart or flush logs
  # /u00/app/mysql/product/mysql-5.0.22/bin/mysqld, Version: 5.0.22-max-log. started with:
  # Tcp port: 3314  Unix socket: /u00/app/mysql/tmp/mysql-5.0.22.sock
  # Time                 Id Command    Argument
  if ( m/^.*mysqld,\sVersion:\s\d\.\d\.[\d]{1,2}.*started with:$/ ) {
    $lWeAreInDmlQueryNow = 0;
    next;
  }
  if ( m/^Tcp port:\s[\d]{4}\s*.*$/ ) {
    $lWeAreInDmlQueryNow = 0;
    next;
  }
  if ( m/^Time\s*Id\s*\Command\s*Argument$/ ) {
    $lWeAreInDmlQueryNow = 0;
    next;
  }

  # 060724 11:00:27       9 Connect     root@localhost on topodb
  #                       9 Query       set autocommit=1
  #                       9 Query       SELECT a.name, r.name AS rock, ...
  #                                     ...
  #                       9 Quit

  # get first line of query
  if ( m/^(\d{6}\s\d{2}:\d{2}:\d{2}|\s)\s*(\d*)\s(Connect|Query|Quit|Init DB|Field List)\s*(.*)$/ ) {

    my $lTimestamp = $1;
    my $lId = $2;
    my $lState = $3;
    my $lCommand = $4;

    # No action on Quit, Init DB, Field List, ...
    if ( $lState eq 'Quit' or $lState eq 'Init DB' or $lState eq 'Field List' ) {
      $lWeAreInDmlQueryNow = 0;
      next;
    }

    # Calulate sleep time
    if ( $oWithThinkTimes ) {
      if ( $lTimestamp =~ m/^\d{6}\s\d{2}:\d{2}:\d{2}$/ ) {

        my ($Yr, $Mon, $Day, $Hr, $Min, $Sec) = ( substr($lTimestamp, 0, 2), substr($lTimestamp, 2, 2), substr($lTimestamp, 4, 2), substr($lTimestamp, 7, 2), substr($lTimestamp,10, 2), substr($lTimestamp,13, 2) );
        $lOldTimestamp = $lNewTimestamp;
        $lNewTimestamp = timelocal($Sec, $Min, $Hr, $Day, $Mon-1, $Yr);
        my $lSleep = $lNewTimestamp - $lOldTimestamp - 1;

        if ( $lSleep >= 1 and $lSleep <= 300 ) {

          if ( $lWriteSemicolon ) {
            print(";\n\n");
          }
          print("/* --with-thinktimes */\n");
          print("/*!50012 SELECT SLEEP($lSleep) */");
          # print "system sleep 3\n";
          $lWriteSemicolon = 1;
        }
      }
    }

    # Connect     root@localhost on topodb
    if ( $lState eq 'Connect' ) {

      $lWeAreInDmlQueryNow = 0;

      if ( $lCommand =~ m/^(\w*)\@(\w*)\son\s(\w*)$/ ) {

        my $lUser = $1;
        my $lHost = $2;
        my $lDatabase = $3;

        if ( $lWriteSemicolon ) {
          print(";\n\n");
        }

        if ( $lDatabase ne '' ) {
          print("use $lDatabase");
          $lWriteSemicolon = 1;
        }
        else {
          $lWriteSemicolon = 0;
        }
      }
      next;
    }

    # Query
    #print $lTimestamp . "\n" . $lId . "\n" . $lState . "\n" . $lCommand . "\n";
    if ( $lCommand =~ m/(insert|replace|update|delete)/i ) {

      if ( $lWriteSemicolon ) {
        print(";\n\n");
      }

      $lWeAreInDmlQueryNow = 1;

      if ( ! $oWithoutDml ) {
        print("$lCommand");
        $lWriteSemicolon = 1;
        next;
      }
      else {
        print("/* $lCommand ...*/\n\n");
        $lWriteSemicolon = 0;
        next;
      }
    }
    else {

      if ( $lWriteSemicolon ) {
        print(";\n\n");
      }

      $lWeAreInDmlQueryNow = 0;

      print("$lCommand");
      $lWriteSemicolon = 1;
      next;
    }
  }

  # get rest of the query
  if ( $lWeAreInDmlQueryNow and $oWithoutDml ) {
    next;
  }
  else {
    print("\n$_");
    next;
  }

  # We should never get here!
  next;
}
if ( $lWriteSemicolon ) {
  print(";\n");
}
close INPUT;

exit(0);

#  time mysql -u root < test.sql > out.txt

# ====================================================================== EOF ===
